/* 
You must link with winmm.lib. If using Visual C++, go to Build->Settings. Flip to the Link page,
and add winmm.lib to the library/object modules.

This app shows how to use the Low Level Wave API to play the WAVE file C:\Windows\Chord.wav.
It also uses the MMIO (RIFF) File Parsing functions.
*/

#include <windows.h>
#include <stdio.h>
#include <conio.h>
#include <mmsystem.h>
#include <malloc.h>




/* Name of the WAVE file */
TCHAR		WaveName[] = "C:\\WINDOWS\\CHORDS.WAV";

/* WAVEFORMATEX structure for reading in the WAVE fmt chunk */
WAVEFORMATEX WaveFormat;

/* file handle for open file */
HMMIO		HMmio;

/* subchunk information structure for reading WAVE file */
MMCKINFO    MmckinfoSubchunk;

/* Count of waveform data still to play */
DWORD		WaveDataPlayed;

/* Handle of opened WAVE Out device */
HWAVEOUT	HWaveOut;

/* The size of each WAVEHDR's wave buffer */
DWORD		WaveBufSize;

/* 2 WAVEHDR structures (for double-buffered playback of WAVE) */
WAVEHDR		WaveHeader[2];

/* Ptr to last WAVEHDR played */
WAVEHDR *	LastHdr;

/* Event used by callback to signal main thread */
HANDLE		Event;





/* ************************** QueueWaveData() *****************************
 * Queues a buffer worth of data for the passed WAVEHDR. This is called
 * by my callback WaveOutProc().
 *
 * Returns a 0 for playback to continue. A -1 if an error. A 1 if the
 * wave has finished playing back and no more buffers need to be queued.
 */

long QueueWaveData(WAVEHDR * waveHeader)
{
	/* More WAVE data to be queued? */
	if (WaveDataPlayed)
	{
		/* Only a partial block left? */
		if (WaveDataPlayed < WaveBufSize)
		{
			/* Tell Windows to read the remaining waveform data into our allocated memory */
			if(mmioRead(HMmio, (HPSTR)waveHeader->lpData, WaveDataPlayed) != (long)WaveDataPlayed) goto bad;

			/* Set the length field to remaining amount */
			waveHeader->dwBufferLength = WaveDataPlayed;

			/* Store last WAVEHDR */
			LastHdr = waveHeader;

			/* Indicate done */
			WaveDataPlayed = 0;

			goto good;
		}

		/* Tell Windows to read another full block of waveform data into our allocated memory */
		if(mmioRead(HMmio, (HPSTR)waveHeader->lpData, WaveBufSize) != (long)WaveBufSize)
		{
bad:		/* Stop playback */
			waveOutPause(HWaveOut);

			/* Oops! */
			printf("ERROR: reading WAVE data!\n");

			return(-1);
		}

		/* Decrease # of bytes yet to play */
		WaveDataPlayed -= WaveBufSize;

good:	/* Clear the WHDR_DONE bit (which the driver set last time that
		   this WAVEHDR was sent via waveOutWrite and was played). Some
		   drivers need this to be cleared */
		waveHeader->dwFlags &= ~WHDR_DONE;
		
		/* Queue the WAVEHDR */
		waveOutWrite(HWaveOut, waveHeader, sizeof(WAVEHDR));

		/* Allow playback to continue */
		return(0);
	}

	/* Did the last WAVEHDR get a chance to play? */
	if (waveHeader == LastHdr)
	{
		/* Finished playing all WAVE data */
		return(1);
	}

	/* Allow playback to continue */
	return(0);
}





/* ************************** WaveOutProc() *****************************
 * My WAVE callback. This is called by Windows every time that some event
 * needs to be handled by my program.
 */

void CALLBACK WaveOutProc(HWAVEOUT waveOut, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2)
{
	/* Has a buffer finished playing? */
	if (uMsg == MM_WOM_DONE)
	{
		/* NOTE: This works upon Windows 95/98/ME, but Windows NT/2000/XP do
		 * not seem to like certain operating system functions called
		 * from within a callback. One such function is waveOutWrite().
		 * If designing your app for Win NT/2000 then you should either
		 * use a different method than a callback handler (such as the
		 * Window Message method), or have your callback signal another
		 * thread to do the below queuing of the next buffer. I recommend
		 * the Windows Message method.
		 */

		/* Queue another buffer */
		if (QueueWaveData((WAVEHDR *)dwParam1))
		{
			/* If no more buffers, notify PlayWave() that the WAVE is done playing */
			SetEvent(Event);
		}
	}
}





/* ************************** PlayWave() *****************************
 * Plays the WAVE. It is presumed that HMmio is an open MultiMedia File
 * handle, currently pointing to the start of the waveform data.
 * WaveDataPlayed is set to the total number of bytes in the waveform.
 * The WAVEFORMATEX structure (ie, WaveFormat) has already been filled
 * in.
 */

void PlayWave() 
{
	DWORD	err;

	/* Open the default WAVE Out Device, specifying my callback */
	if ((err = waveOutOpen(&HWaveOut, WAVE_MAPPER, &WaveFormat, (DWORD)WaveOutProc, 0, CALLBACK_FUNCTION)))
	{
		printf("ERROR: Can't open WAVE Out Device! -- %08X\n", err);
		return;
	}

	/* I'll allocate 2 buffers large enough to hold 2 seconds worth of waveform data. NOTE: Just to make it
	   easy, I'll use 1 call to VirtualAlloc to allocate both buffers, but you can use 2 separate calls since
	   the buffers do NOT need to be contiguous */
	WaveBufSize = WaveFormat.nAvgBytesPerSec << 1;

	if (!(WaveHeader[0].lpData = (char *)VirtualAlloc(0, WaveBufSize<<1, MEM_COMMIT, PAGE_READWRITE)))
	{
		printf("ERROR: Can't allocate memory for WAVE buffer!\n");
		goto bad1;

	}

	/* Fill in WAVEHDR fields for buffer starting address and size */
	WaveHeader[1].lpData = WaveHeader[0].lpData + WaveBufSize;
	WaveHeader[1].dwBufferLength = WaveHeader[0].dwBufferLength = WaveBufSize;

	/* Leave other WAVEHDR fields at 0 */

	/* Prepare the 2 WAVEHDR's */
	if ((err = waveOutPrepareHeader(HWaveOut, &WaveHeader[0], sizeof(WAVEHDR))))
	{
		printf("ERROR: preparing WAVEHDR 1! -- %08X\n", err);
		goto bad2;
	}

	if ((err = waveOutPrepareHeader(HWaveOut, &WaveHeader[1], sizeof(WAVEHDR))))
	{		
		printf("ERROR: preparing WAVEHDR 2! -- %08X\n", err);
		goto bad3;
	}

	/* Queue first buffer (it starts playing because device isn't paused) */
	if (!QueueWaveData(&WaveHeader[0]))
	{
		/* Queue second buffer */
		if (QueueWaveData(&WaveHeader[1]) >= 0)
		{
			/* Wait for playback to finish. My callback notifies me when all wave data has been played */
			WaitForSingleObject(Event, INFINITE);
		}
	}

	/* Stop Windows queuing of buffers (and calling of my callback) */
	waveOutReset(HWaveOut);

	/* Unprepare WAVE buffers */
	waveOutUnprepareHeader(HWaveOut, &WaveHeader[1], sizeof(WAVEHDR));
bad3:
	waveOutUnprepareHeader(HWaveOut, &WaveHeader[0], sizeof(WAVEHDR));

bad2:
	/* Free WAVE buffers */
	VirtualFree(WaveHeader[0].lpData, WaveBufSize<<1, MEM_FREE);

bad1:
	/* Close WAVE Out device */
	waveOutClose(HWaveOut);
}





/* ******************************** main() ******************************** */

int main(int argc, char **argv)
{
	MMCKINFO	mmckinfoParent;		/* parent chunk information structure */

	/* Allocate an Event signal */
	if ((Event = CreateEvent(0, FALSE, FALSE, 0)))
	{
		/* Open the file for reading */
		if ((HMmio = mmioOpen(&WaveName[0], 0, MMIO_READ)))
		{
			/*	Tell Windows to locate a WAVE FileType chunk header somewhere in the file.
				This marks the start of any embedded WAVE format within the file */
			mmckinfoParent.fccType = mmioFOURCC('W', 'A', 'V', 'E'); 
			if (mmioDescend(HMmio, (LPMMCKINFO)&mmckinfoParent, 0, MMIO_FINDRIFF)) 
			{
				/* Oops! No embedded WAVE format within this file */
				printf("ERROR: This file doesn't contain a WAVE!\n");

				/* Close the file and exit with error */
out:			mmioClose(HMmio, 0);

				goto out2;
			} 
 
			/* Tell Windows to locate the WAVE's "fmt " chunk, and read in its size field */
			MmckinfoSubchunk.ckid = mmioFOURCC('f', 'm', 't', ' '); 
			if (mmioDescend(HMmio, &MmckinfoSubchunk, &mmckinfoParent, MMIO_FINDCHUNK)) 
			{
				/* Oops! The required fmt chunk was not found! */
				printf("ERROR: Required fmt chunk was not found!\n");
				goto out;
			}
 
			/* Tell Windows to read in the "fmt " chunk into our WAVEFORMATEX structure */
			if (mmioRead(HMmio, (HPSTR)&WaveFormat, MmckinfoSubchunk.cksize) != (LRESULT)MmckinfoSubchunk.cksize)
			{
				/* Oops! */
				printf("ERROR: reading the fmt chunk!\n");
				goto out;
			}
 
			/*	Ascend out of the "fmt " subchunk. If you plan to parse any other chunks in the file, you need to
				"ascend" out of any chunk that you've mmioDescend()'ed into */
			mmioAscend(HMmio, &MmckinfoSubchunk, 0); 

			/*	Tell Windows to locate the data chunk. Upon return, the file
				pointer will be ready to read in the actual waveform data within
				the data chunk */
			MmckinfoSubchunk.ckid = mmioFOURCC('d', 'a', 't', 'a'); 
			if (mmioDescend(HMmio, &MmckinfoSubchunk, &mmckinfoParent, MMIO_FINDCHUNK)) 
			{
				/* Oops! */
				printf("ERROR: reading the data chunk!\n");
				goto out;
			}
 
			/* Set how many total bytes to play */
			WaveDataPlayed = MmckinfoSubchunk.cksize;
		
			/* Play the wave */
			PlayWave();

			/* Close the file */
			mmioClose(HMmio, 0);

			/* Free the Event */
			CloseHandle(Event);

			/* Success */
			return(0);
		}

		/* Oops! */
		printf("ERROR: Can't find the WAVE file!\n");

		/* Free the Event */
out2:	CloseHandle(Event);

		return(-2);
	}

	/* Oops! */
	printf("ERROR: Can't allocate Event Semaphore!\n");
	
	return(-1);
}
