/*
You must link with winmm.lib. If using Visual C++, go to Build->Settings. Flip to the Link page,
and add winmm.lib to the library/object modules.

This app plays a musical phrase from the song "Twinkle, Twinkle Little Star". It sends the MIDI
events to the default MIDI Out device using the Stream API. The Stream API does the actual timing
out of each MIDI event. Therefore we don't have to use a multi-media timer nor delays.

I incorporate a callback into this example.
*/

#include <windows.h>
#include <stdio.h>
#include <conio.h>
#include <mmsystem.h>





/* Here's an array of MIDI Note-On and Note-Off events to play our musical phrase. We declare the
array as an array of unsigned longs. Makes it easier to statically declare the MIDIEVENT structures
in an array.

I've set the timing fields in terms of PPQN clocks, as they would be found in MIDI File using
96 PPQN Time Division.

NOTES: The first event is a System Exclusive event to set Master Volume to full. Note that the high
byte of the dwEvent field (ie, the 3rd unsigned long) must be MEVT_LONGMSG, and the remaining bytes
must be the count of how many SysEx bytes follow. Also note that the data must be padded out to a
doubleword boundary. That's implicit in our declaration of this as an unsigned long array.

The second event is a Tempo event which sets tempo to 500,000 microseconds per quarter
note. I put this event at the start of the array so that I don't have to use midiStreamProperty()
to set tempo prior to playback. Note that the high byte of dwEvent must be MEVT_TEMPO to tell the
stream device that this is a Tempo event.

Also, note that the MIDI event is packed into a long the same way that it would be to send to
midiOutShortMsg().

I'm implementing a callback in this program. Whenever I set the MEVT_F_CALLBACK flag of some
event, my callback will be called. There are two purposes for my callback. First, I want to
implement a visual metronome that displays the count off of quarter notes in 4\4 time. Secondly,
I want my callback to signal my main() function when playback is all done.

For the purposes of the metronome, I'm going to use the NOP events to signal my callback. I'm
going to place a NOP event at a time when each quarter note should occur. For these NOP events,
I've arbitrary chosen to set lowest byte of the dwEvent field to 1 for a beat that occurs
upon a measure boundary (ie, the downbeat of a measure) or a 0 if just an ordinary beat (within
the measure). I assume 4\4 time.

For the purposes of knowing when the playback is done, Windows automatically calls my callback
when it finishes playing a MIDIHDR's block of events.
*/

unsigned long Phrase[] = {0, 0, ((unsigned long)MEVT_LONGMSG<<24) | 8, 0x047F7FF0, 0xF77F7F01,
0, 0, ((unsigned long)MEVT_TEMPO<<24) | 0x0007A120,
0, 0, 0x007F3C90,
0, 0, 0x02000001 | MEVT_F_CALLBACK,	/* A NOP event to mark the downbeat of the first measure */
48, 0, 0x00003C90,
0, 0, 0x007F3C90,
48, 0, 0x00003C90,
0, 0, 0x02000000 | MEVT_F_CALLBACK,	/* A NOP event to mark the 2nd beat of the first measure */

0, 0, 0x007F4390,
48, 0, 0x00004390,
0, 0, 0x007F4390,
48, 0, 0x00004390,
0, 0, 0x02000000 | MEVT_F_CALLBACK,

0, 0, 0x007F4590,
48, 0, 0x00004590,
0, 0, 0x007F4590,
48, 0, 0x00004590,
0, 0, 0x02000000 | MEVT_F_CALLBACK,

0, 0, 0x007F4390,
86, 0, 0x00004390,

10, 0, 0x007F4190,
0, 0, 0x02000001 | MEVT_F_CALLBACK,
48, 0, 0x00004190,
0, 0, 0x007F4190,
48, 0, 0x00004190,
0, 0, 0x02000000 | MEVT_F_CALLBACK,

0, 0, 0x007F4090,
48, 0, 0x00004090,
0, 0, 0x007F4090,
48, 0, 0x00004090,
0, 0, 0x02000000 | MEVT_F_CALLBACK,

0, 0, 0x007F3E90,
48, 0, 0x00003E90,
0, 0, 0x007F3E90,
48, 0, 0x00003E90,
0, 0, 0x02000000 | MEVT_F_CALLBACK,

0, 0, 0x007F3C90,
96, 0, 0x00003C90};


HANDLE	Event;
DWORD	Measure = 0;
DWORD	Beat = 0;





/*************************** midiCallback() *****************************
 * Here's my Stream callback that Windows' Stream API (ie, midiStreamOut)
 * calls whenever 1 of 4 possible things happen:
 *
 * 1).	I open a Stream device via midiStreamOpen(). In this case, the
 *		uMsg arg to my callback will be MOM_OPEN. The handle arg will
 *		be the same as what is returned to midiStreamOpen(). The
 *		dwInstance arg is whatever I passed to midiStreamOpen() as its
 *		dwInstance arg.
 *
 * 2).	I close a Stream Device via midiStreamClose(). In this case, the
 *		uMsg arg to my callback will be MOM_CLOSE. The handle arg will
 *		be the same as what was passed to midiStreamClose(). The
 *		dwInstance arg is whatever I passed to midiStreamOpen() as its
 *		dwInstance arg when I initially opened this handle.
 *
 * 3).	midiStreamOut encounters an event that has its MEVT_F_CALLBACK flag
 *		set. In this case, the uMsg arg to my callback will be MOM_POSITIONCB.
 *		The handle arg will be the same as what is passed to midiStreamOut().
 *		The dwInstance arg is whatever I passed to midiStreamOpen() as its
 *		dwInstance arg when I initially opened this handle. The dwParam1 arg
 *		points to the MIDIHDR that I passed to midiStreamOut(). The dwOffset
 *		field of the MIDIHDR will indicate the byte offset into the buffer
 *		for the event that caused this callback to be called.
 *
 * 4).	midiStreamOut finishes playing a MIDIHDR's block of data.
 *		In this case, the uMsg arg to my callback will be MOM_DONE.
 *		The handle arg will be the same as what is passed to midiStreamOut().
 *		The dwInstance arg is whatever I passed to midiStreamOpen() as its
 *		dwInstance arg when I initially opened this handle. The dwParam1 arg
 *		points to the MIDIHDR that I passed to midiStreamOut().
 *
 * NOTE: The dwParam2 arg is not used. This is reserved for future use.
 *  
 * There are two purposes for my callback here. First, I want to implement a
 * visual metronome that displays the count off of quarter notes in 4\4 time. Secondly,
 * I want my callback to signal my main() function when playback is all done.
 *
 * For the purposes of the metronome, I use the NOP events to signal my callback. I
 * placed a NOP event at a time when each quarter note should occur. For these NOP events,
 * I arbitrary chose to set lowest byte of the dwEvent field to 1 for a beat that occurs
 * upon a measure boundary (ie, the downbeat of a measure) or a 0 if just an ordinary
 * beat (within the measure). I assume 4\4 time. I simply write the metronome measure
 * and beat to the console window.
 *
 * For the purposes of knowing when the playback is done, Windows automatically calls
 * my callback when it finishes playing a MIDIHDR's block of events. At this point,
 * I use the signal event that main() created to wake main() up.
 *************************************************************************/

void CALLBACK midiCallback(HMIDIOUT handle, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2)
{
    LPMIDIHDR	lpMIDIHeader;
    MIDIEVENT *	lpMIDIEvent;
	TCHAR		buffer[80];

	/* Determine why Windows called me */
	switch (uMsg)
	{
		/* Got some event with its MEVT_F_CALLBACK flag set */
		case MOM_POSITIONCB:

			/*	Assign address of MIDIHDR to a LPMIDIHDR variable. Makes it easier to access the
				field that contains the pointer to our block of MIDI events */
			lpMIDIHeader = (LPMIDIHDR)dwParam1;

			/* Get address of the MIDI event that caused this call */
			lpMIDIEvent = (MIDIEVENT *)&(lpMIDIHeader->lpData[lpMIDIHeader->dwOffset]);

			/*	Normally, if you had several different types of events with the
				MEVT_F_CALLBACK flag set, you'd likely now do a switch on the highest
				byte of the dwEvent field, assuming that you need to do different
				things for different types of events. But here, I know that all of my
				events happen to be NOP's
			*/

			/*	Check the low byte of the dwEvent field. If a 1, then this is a measure
				downbeat. In that case, we reset the Beat variable to 1, and increment
				the Measure variable. Otherwise, it must be a regular beat, in which
				case we just increment the Beat variable
			*/
			if (lpMIDIEvent->dwEvent & 0x00000001)
			{
				Measure++;
				Beat = 0;
			}
			Beat++;

			/* Display Measure and Beat */
			sprintf(&buffer[0], "Measure = %-3u Beat = %-1u\r", Measure, Beat);
			_cputs(&buffer[0]);

			break;

		/* The last event in the MIDIHDR has played */
		case MOM_DONE:

			/* Wake up main() */
			SetEvent(Event);

			break;


		/* Process these messages if you desire */
/*
		case MOM_OPEN:
        case MOM_CLOSE:

			break;
*/
  }
}




/*********************** PrintMidiOutErrorMsg() **************************
 * Retrieves and displays an error message for the passed MIDI Out error
 * number. It does this using midiOutGetErrorText().
 *************************************************************************/

void PrintMidiOutErrorMsg(unsigned long err)
{
#define BUFFERSIZE 120
	char	buffer[BUFFERSIZE];
	
	if (!(err = midiOutGetErrorText(err, &buffer[0], BUFFERSIZE)))
	{
		printf("%s\r\n", &buffer[0]);
	}
	else if (err == MMSYSERR_BADERRNUM)
	{
		printf("Strange error number returned!\r\n");
	}
	else
	{
		printf("Specified pointer is invalid!\r\n");
	}
}





/* ******************************** main() ******************************** */

int main(int argc, char **argv)
{
	HMIDISTRM		handle;
	MIDIHDR			midiHdr;
	unsigned long	err;
	MIDIPROPTIMEDIV prop;

	/* Allocate an Event signal for my callback to wake me up */
	if ((Event = CreateEvent(0, FALSE, FALSE, 0)))
	{
		/* Open default MIDI Out stream device */
		err = 0;
		if (!(err = midiStreamOpen(&handle, &err, 1, (DWORD)midiCallback, 0, CALLBACK_FUNCTION)))
		{
			/* Set the stream device's Time Division to 96 PPQN */
			prop.cbStruct = sizeof(MIDIPROPTIMEDIV);
			prop.dwTimeDiv = 96;
			err = midiStreamProperty(handle, (LPBYTE)&prop, MIDIPROP_SET|MIDIPROP_TIMEDIV);
			if (err)
			{
				PrintMidiOutErrorMsg(err);
			}

			/* Store pointer to our stream (ie, buffer) of messages in MIDIHDR */
			midiHdr.lpData = (LPBYTE)&Phrase[0];

			/* Store its size in the MIDIHDR */
			midiHdr.dwBufferLength = midiHdr.dwBytesRecorded = sizeof(Phrase);

			/* Flags must be set to 0 */
			midiHdr.dwFlags = 0;

			/* Prepare the buffer and MIDIHDR */
			err = midiOutPrepareHeader(handle,  &midiHdr, sizeof(MIDIHDR));
			if (!err)
			{
		        /* Queue the Stream of messages. Output doesn't actually start
				until we later call midiStreamRestart().
				*/
				err = midiStreamOut(handle, &midiHdr, sizeof(MIDIHDR));
				if (err)
				{
bad:				PrintMidiOutErrorMsg(err);
					midiOutUnprepareHeader(handle, &midiHdr, sizeof(MIDIHDR));
				}
				else
				{
					/* Start outputting the Stream of messages. This should return immediately
					as the stream device will time out and output the messages on its own in
					the background.
					*/
				    err = midiStreamRestart(handle);
					if (err)
					{
						goto bad;
					}

					/* Wait for playback to stop. Windows signals me using that Event I created */
					WaitForSingleObject(Event, INFINITE);

					/* Unprepare the buffer and MIDIHDR */
					midiOutUnprepareHeader(handle, &midiHdr, sizeof(MIDIHDR));
				}
			}

			/* An error. Print a message */
			else
			{
				PrintMidiOutErrorMsg(err);
			}
	
			 /* Close the MIDI Stream */
			midiStreamClose(handle);
		}
		else
		{
			printf("Error opening the default MIDI Stream!\r\n");
			PrintMidiOutErrorMsg(err);
		}

		/* Free the Event */
		CloseHandle(Event);
	}
	else
	{
		printf("Error creating Event signal!\r\n");
	}

	return(0);
}