/*
This sends a MIDI System Exclusive message to the default MIDI Output device, The
message sets the Master Volume. Not all output devices support this System
Exclusive message. It depends upon the device's MIDI implementation.

The volume is input by the user which we then break up into
2 7-bit bytes in the System Exclusive message.
*/

#include <windows.h>
#include <stdio.h>
#include <conio.h>
#include <mmsystem.h>





unsigned char	Vol[] = {0xF0, 0x7F, 0x7F, 0x04, 0x01, 0x7F, 0x7F, 0xF7};





/*********************** PrintMidiOutErrorMsg() **************************
 * Retrieves and displays an error message for the passed MIDI Out error
 * number. It does this using midiOutGetErrorText().
 *************************************************************************/

void PrintMidiOutErrorMsg(unsigned long err)
{
#define BUFFERSIZE 120
	char	buffer[BUFFERSIZE];
	
	if (!(err = midiOutGetErrorText(err, &buffer[0], BUFFERSIZE)))
	{
		printf("%s\r\n", &buffer[0]);
	}
	else if (err == MMSYSERR_BADERRNUM)
	{
		printf("Strange error number returned!\r\n");
	}
	else
	{
		printf("Specified pointer is invalid!\r\n");
	}
}





/********************************** main() *******************************
 *************************************************************************/

int main(int argc, char **argv)
{
	HMIDIOUT		handle;
	MIDIHDR			midiHdr;
	UINT			err;
	char			ch;

	/* Open default MIDI Out device */
	if (!(err = midiOutOpen(&handle, 0, 0, 0, CALLBACK_NULL)))
	{
		/* Get the desired volume from user */
		err = 0;
		_cputs("Input desired volume (0 to 16,383), followed by ENTER:\r\n");
		do
		{
			ch = _getche();
			if (ch >= '0' && ch <= '9')
			{
				err = (err * 10) + (ch - '0');
			}
		} while(ch != 10 && ch != 13);

		if (err > 16383)
		{
			printf("Volume can't exceed 16,383! Defaulting to maximum volume...\r\n");
			err = 16383;
		}

		/* Put it into the System Exclusive message */
		Vol[5] = (unsigned char)(err & 0x007F);
		Vol[6] = (unsigned char)(err >> 7);

	    /* Store pointer in MIDIHDR */
		midiHdr.lpData = (LPBYTE)&Vol[0];

		/* Store its size in the MIDIHDR */
		midiHdr.dwBufferLength = sizeof(Vol);

		/* Flags must be set to 0 */
		midiHdr.dwFlags = 0;

		/* Prepare the buffer and MIDIHDR */
        err = midiOutPrepareHeader(handle,  &midiHdr, sizeof(MIDIHDR));
	    if (!err)
		{
	        /* Output the SysEx message. Note that this could return immediately if
			the device driver can output the message on its own in the background.
			Otherwise, the driver may make us wait in this call until the entire
			data is output
			*/
		    err = midiOutLongMsg(handle, &midiHdr, sizeof(MIDIHDR));
			if (err)
			{
				PrintMidiOutErrorMsg(err);
			}

			/* Unprepare the buffer and MIDIHDR */
			while (MIDIERR_STILLPLAYING == midiOutUnprepareHeader(handle, &midiHdr, sizeof(MIDIHDR)))
			{
		        /* Delay to give it time to finish */
				Sleep(1000);
			}
		}

		/* An error. Print a message */
		else
		{
			PrintMidiOutErrorMsg(err);
		}

		/* Close the MIDI device */
		midiOutClose(handle);
	}
	else
	{
		printf("Error opening default MIDI Out device!\r\n");
		PrintMidiOutErrorMsg(err);
	}

	return(0);
}
