/* Checksum -- Calculates the checksum for a Roland System Exclusive message.
 *
 * This presents a window into which the user can type the 3 or 4 address bytes,
 * and data byte(s) that will appear in a "Data Set 1" Roland System Exclusive
 * message. It will then calculate and display the appropriate checksum for that
 * message. It can also calculate the checksum for a "Data Request 1".
 */

/* Windows include files */
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>

/* Here, I include #define's for my symbols used in my RESOURCE (.RC) and SOURCE (.C) files */
#include "resource.h"




/* *********************** GLOBALS ************************ */

/* Executable handle */
HINSTANCE		MyInstance;

/* My main window handle */
HWND			MainWindow = 0;

/* Strings for combobox */
const char TypeStrings[] = {"Decimal\0\
Hex\0\
Binary\0\
\0"};

/* Error messages */
const char			BadWindow[] = "Can't create window!";
const char			ErrorStr[] = "ERROR";








/***************************** asciiToNum() *****************************
 * Converts the ascii string of digits (expressed in base 10) to a 32-bit
 * long. Returns the end of the ascii string of digits (ie, after the last
 * non-numeric digit) in passed handle (unless that handle is 0). This is
 * useful if you want to parse several numeric values from a string
 * containing several -- by calling this function for each value, and
 * passing the handle that is returned, for the buf arg in the next call.
 ************************************************************************/

long asciiToNum(unsigned char *buf, unsigned char **next)
{
	unsigned char *	start;
	unsigned long	result = 0;
	unsigned long	extra;
	unsigned char	chr;

	/* Skip leading spaces */
	start = buf;

	/* A negative value? */
	if (*buf == '-') ++buf;

	/* Convert next digit */
	while (*buf)
	{
		chr = *buf - '0';
		if (chr > 9) break;
		extra = result << 3;
		result += (result + extra + chr);
		buf++;
	}

	/* Store end of numeric digits */
	if (next) *next = buf;

	/* Return value */
	if (*start != '-') return((long)result);
	return(-result);
}





/**************************** asciiToNum2() *****************************
 * Converts the ascii string of digits (expressed in base 10, or base 16
 * if format is 'H', or base 2 if format is 'B') to a 32-bit long. Returns
 * the end of the ascii string of digits (ie, after the last non-numeric
 * digit) in passed handle (unless that handle is 0). This is useful if
 * you want to parse several numeric values from a string containing
 * several -- by calling this function for each value, and passing the
 * handle that is returned, for the buf arg in the next call.
 ************************************************************************/

long asciiToNum2(unsigned char *buf, unsigned char **next, unsigned char format)
{
	unsigned long	result = 0;
	unsigned char	chr;

	if (format != 'H' && format != 'B') return(asciiToNum(buf, next));

	/* Convert next digit */
	while (*buf)
	{
		if (format == 'H')
		{
			chr = *buf;
			if (chr >= 'a') chr &= 0x5F;
			chr -= '0';
			if (chr >= 17 && chr <= 22) chr -= 7;
			if (chr > 15) break;
			result <<= 4;
		}
		else
		{
			chr = *buf - '0';
			if (chr > 1) break;
			result <<= 1;
		}
		result |= chr;
		buf++;
	}

	/* Store end of numeric digits */
	if (next) *next = buf;

	/* Return value */
	return((long)result);
}





/***************************** hexToAscii() ******************************
 * Converts the passed value to a null-terminated ascii string of digits in
 * base 16, putting that string into the passed buffer. Returns the end of
 * the ascii string (ie, pointer to final null byte), which is useful for
 * appending something else to the buffer, or getting the strlen by
 * subtracting a pointer to the buffer's head from the return.
 *************************************************************************/

const unsigned char * hexToAscii(unsigned long num, unsigned char *buf, unsigned char size)
{
	unsigned char * ptr;
	unsigned char	chr;

	size <<= 1;
	buf += size;
	ptr = buf;
	*(buf) = 0;
	while (size--)
	{
		ptr--;
		chr = (unsigned char)(num & 0x0000000F) + '0';
		if (chr > '9') chr += ('A'-':');
		num >>= 4;
		*(ptr) = chr;
	}

	return(buf);
}





/************************* fileSkipSpaces() ***************************
 * Skips leading blanks in the passed string, and returns a pointer to
 * the first non-blank char.
 **********************************************************************/

unsigned char * skipSpaces(const unsigned char * str)
{
	while (*str == ' ' || *str == 0x09) ++str;
	return((unsigned char *)str);
}





/*************************** putStrings() *****************************
 * Fills a COMBOBOX with the strings "Decimal", "Hex", and "Binary",
 * and selects one initially.
 **********************************************************************/

void putStrings(HWND hwnd, unsigned long select)
{
	const char		*ptr;
	RECT			rc;

	ptr = &TypeStrings[0];
	do
	{
		SendMessage(hwnd, CB_ADDSTRING, 0, (LPARAM)ptr);
		ptr += lstrlen(ptr) + 1;
	} while (*ptr);

	SendMessage(hwnd, CB_SETCURSEL, select, 0);

	/* Now that the items are added, figure out how much we need to increase
	 * the height of the control to allow the listbox to "drop down"
	 */
	GetWindowRect(hwnd, &rc);
	SetWindowPos(hwnd, 0, 0, 0, rc.right - rc.left, (rc.bottom - rc.top) << 2, SWP_NOMOVE|SWP_NOZORDER);
}




/******************************* mainWndProc() *********************************
 * This is my message procedure for MainWindow. It is called by Windows whenever
 * there is a message for my window to process.
 *******************************************************************************/

long APIENTRY mainWndProc(HWND hwnd, UINT uMsg, UINT wParam, long lParam)
{
	switch(uMsg)
	{
		/* ***************************************************************
		   ======================== Menu or Buttons ====================== */
		case WM_COMMAND:
		{
			switch (wParam)
			{
				/* Calculate Roland checksum */
				case IDC_CALCULATE:
				{
					char			buf[256];
					char			*ptr;
					unsigned long	sum;
					char			type;

					/* Initially sum is 0 */
					sum = 0;

					/* See if the user wants the address string to be regarded as hex, decimal, or binary */
					type = (char)SendDlgItemMessage(hwnd, IDC_ADDRESSTYPE, CB_GETCURSEL, 0, 0);
					switch (type)
					{
						case 2:
						{
							type = 'B';
							break;
						}

						case 1:
						{
							type = 'H';
							break;
						}

						default:
						{
							type = 'D';
						}
					}

					/* Get the address string */
					GetDlgItemText(hwnd, IDC_ADDRESS, &buf[0], 256);

					/* Convert the ascii string into one sum */
					ptr = skipSpaces(&buf[0]);
					while (*ptr)
					{
						sum += asciiToNum2(ptr, &ptr, type);
						ptr = skipSpaces(ptr);
					}

					/* See if the user wants the data string to be regarded as hex, decimal, or binary */
					type = (char)SendDlgItemMessage(hwnd, IDC_DATATYPE, CB_GETCURSEL, 0, 0);
					switch (type)
					{
						case 2:
						{
							type = 'B';
							break;
						}

						case 1:
						{
							type = 'H';
							break;
						}

						default:
						{
							type = 'D';
						}
					}

					/* Get the data string */
					GetDlgItemText(hwnd, IDC_DATA, &buf[0], 256);

					/* Convert the ascii string into one sum, and add to previous sum */
					ptr = skipSpaces(&buf[0]);
					while (*ptr)
					{
						sum += asciiToNum2(ptr, &ptr, type);
						ptr = skipSpaces(ptr);
					}

					/* Finish calculating Roland checksum by making sure that the sum is less
					 * than 128 (ie, we just mask off bit 7 instead of dividing by 128 and
					 * taking the remainder -- it's faster), then subtracting from 128, and
					 * finally adjusting for 128=0 (again, by masking off bit 7)
					 */
					sum = (128 - (sum & 0x7F)) & 0x7F;

					/* Display checksum as one hex byte */
					hexToAscii(sum, &buf[0], 1);
					SetDlgItemText(hwnd, IDC_CHECKSUM, &buf[0]);

					break;
				}

				default:
				{
					/* Blank the Checksum display if any of the other controls may have been changed */
					SetDlgItemText(hwnd, IDC_CHECKSUM, " ");
				}
			}

			break;
		}

		/* ****************************************************************** */
		/* ====================== Create main window ======================== */
		case WM_INITDIALOG:
		{
			HICON	icon;

			/* Load/set icon for System menu on the window. I put my icon
			 * in this executable's resources. Note that windows frees this
			 * when my window is closed.
			 */
			if ((icon = LoadIcon(MyInstance, MAKEINTRESOURCE(IDI_MAIN_ICON))))
				SetClassLong(hwnd, GCL_HICON, (LONG)icon);

			/* Set IDC_ADDRESS limit to 20 chars */
			SendDlgItemMessage(hwnd, IDC_ADDRESS, EM_SETLIMITTEXT, (WPARAM)20, 0);

			/* Set IDC_DATA limit to 256 chars. If a user needs to calculate a
			 * checksum for a Data Set 1 containing more than about 80 data
			 * bytes, then this program would need to allocate a buffer for
			 * GetDlgItemText() instead of limiting it to 256
			 */
			SendDlgItemMessage(hwnd, IDC_DATA, EM_SETLIMITTEXT, (WPARAM)256, 0);

			/* Put the strings "Decimal", "Hex", and "Binary" in IDC_ADDRESSTYPE,
			 * and initially select Hex
			 */
			putStrings(GetDlgItem(hwnd, IDC_ADDRESSTYPE), 1);

			/* Repeat for IDC_DATATYPE, but initially select Decimal */
			putStrings(GetDlgItem(hwnd, IDC_DATATYPE), 0);

			return(1);
		}

		/* ****************************************************************** */
		/* =================== User wants to close window =================== */
		case WM_CLOSE:
		{
			/* Close this window */
			DestroyWindow(hwnd);

			return(1);
		}

		case WM_DESTROY:
		{
 			/* Post the WM_QUIT message to quit the message loop in WinMain() */
			PostQuitMessage(0);

			return(1);
		}
	}

	/* Indicate that I didn't handle the msg */
	return(0);
} 





/******************************* WinMain() ********************************
 * Program Entry point
 **************************************************************************/

int WINAPI WinMain(HINSTANCE hinstExe, HINSTANCE hinstPrev, LPSTR lpszCmdLine, int nCmdShow)
{
	MSG		msg;

	/* Save Instance handle which I need when opening the window (see WM_INITDIALOG) */
	MyInstance = hinstExe;

	/* Loads Windows common control's DLL  */
	InitCommonControls();

	/* Create Main window */
	if (!(MainWindow = CreateDialog(MyInstance, MAKEINTRESOURCE(IDD_MAINWINDOW), 0, mainWndProc)))
	{
		MessageBox(0, &BadWindow[0], &ErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
		return(-1);
	}

	/* Show the window with default size/position */
	ShowWindow(MainWindow, SW_SHOWDEFAULT);
	UpdateWindow(MainWindow);

	/* Here's my message loop which is executed until the user closes
	 * down my main window.
	 */
	
	/* Get the next msg (until WM_QUIT) */
	while (GetMessage(&msg, 0, 0, 0) == 1)
	{
		/* Translate WM_KEYDOWN for edit controls, handle TAB */
		if (!IsDialogMessage(MainWindow, &msg))

			/* Send msg to window procedure */
			DispatchMessage(&msg);
	}

	/* Exit */
	return(0);
}
